﻿//+------------------------------------------------------------------+
//|                                 EconomicCalendar.mq5             |
//|                        Copyright 2024, YourName                  |
//+------------------------------------------------------------------

#property version   "1.00"
#property description "Economic Calendar with News Details"
#property indicator_chart_window
#property indicator_plots 0

//+------------------------------------------------------------------+
//| Input parameters                                                 |
//+------------------------------------------------------------------+
input int      RefreshInterval = 300;           // Refresh interval (seconds)
input bool     ShowHighImpactOnly = false;      // Show only high impact events
input int      HoursAhead = 48;                 // Show events X hours ahead
input color    ColorHighImpact = clrRed;        // High impact color
input color    ColorMediumImpact = clrOrange;   // Medium impact color
input color    ColorLowImpact = clrYellow;      // Low impact color
input color    ColorPositive = clrLime;         // Positive surprise color
input color    ColorNegative = clrRed;          // Negative surprise color
input int      PanelWidth = 700;                // Panel width
input int      PanelHeight = 400;               // Panel height

//+------------------------------------------------------------------+
//| Global variables                                                 |
//+------------------------------------------------------------------+
string         BaseCurrency;
string         QuoteCurrency;
datetime       LastRefreshTime = 0;
int            TotalEvents = 0;
int            PendingEvents = 0;
bool           PanelInitialized = false;
int            PanelX = 20;
int            PanelY = 50;
bool           DetailsPanelVisible = false;
int            SelectedEventIndex = -1;

// Event structure
struct EconomicEvent
  {
   datetime          EventTime;
   string            Currency;
   string            Title;
   string            Impact;
   double            Previous;
   double            Forecast;
   double            Actual;
   string            Deviation;
   double            Surprise;
   bool              IsLive;
   string            Description;
  };

EconomicEvent EventsArray[];
bool          IsConnected = false;

//+------------------------------------------------------------------+
//| Generate realistic demo data for specific pair                  |
//+------------------------------------------------------------------+
void GenerateDemoData()
  {
   ArrayResize(EventsArray, 8);

// Extract currencies
   string symbol = Symbol();
   if(StringLen(symbol) >= 6)
     {
      BaseCurrency = StringSubstr(symbol, 0, 3);
      QuoteCurrency = StringSubstr(symbol, 3, 3);
     }
   else
     {
      BaseCurrency = "EUR";
      QuoteCurrency = "USD";
     }

// Event data
   string eventTitles[] =
     {
      "Interest Rate Decision",
      "CPI (Consumer Price Index)",
      "Non-Farm Payrolls",
      "Unemployment Rate",
      "GDP (Gross Domestic Product)",
      "Retail Sales",
      "Manufacturing PMI",
      "Trade Balance"
     };

   string impacts[] = {"High","Medium","Low"};

   int eventIndex = 0;
   datetime currentTime = TimeCurrent();

// Generate events for base currency
   for(int i=0; i<4 && eventIndex<8; i++)
     {
      EventsArray[eventIndex].EventTime = currentTime + (i * 3600 * 3);
      EventsArray[eventIndex].Currency = BaseCurrency;
      EventsArray[eventIndex].Title = eventTitles[i];
      EventsArray[eventIndex].Impact = impacts[i % 3];
      EventsArray[eventIndex].Previous = 2.5 + (MathRand() % 50) * 0.1;
      EventsArray[eventIndex].Forecast = EventsArray[eventIndex].Previous + (MathRand() % 10 - 5) * 0.1;
      EventsArray[eventIndex].Actual = EventsArray[eventIndex].Forecast + (MathRand() % 6 - 3) * 0.1;
      EventsArray[eventIndex].IsLive = false;
      EventsArray[eventIndex].Description = GenerateEventDescription(
            EventsArray[eventIndex].Currency,
            EventsArray[eventIndex].Title,
            EventsArray[eventIndex].Previous,
            EventsArray[eventIndex].Forecast,
            EventsArray[eventIndex].Actual
                                            );
      eventIndex++;
     }

// Generate events for quote currency
   for(int i=0; i<4 && eventIndex<8; i++)
     {
      EventsArray[eventIndex].EventTime = currentTime + (i * 3600 * 3) + 3600;
      EventsArray[eventIndex].Currency = QuoteCurrency;
      EventsArray[eventIndex].Title = eventTitles[i+4];
      EventsArray[eventIndex].Impact = impacts[(i+1) % 3];
      EventsArray[eventIndex].Previous = 2.5 + (MathRand() % 50) * 0.1;
      EventsArray[eventIndex].Forecast = EventsArray[eventIndex].Previous + (MathRand() % 10 - 5) * 0.1;
      EventsArray[eventIndex].Actual = EventsArray[eventIndex].Forecast + (MathRand() % 6 - 3) * 0.1;
      EventsArray[eventIndex].IsLive = false;
      EventsArray[eventIndex].Description = GenerateEventDescription(
            EventsArray[eventIndex].Currency,
            EventsArray[eventIndex].Title,
            EventsArray[eventIndex].Previous,
            EventsArray[eventIndex].Forecast,
            EventsArray[eventIndex].Actual
                                            );
      eventIndex++;
     }

// Calculate deviation and surprise
   for(int i=0; i<eventIndex; i++)
     {
      if(EventsArray[i].Forecast != 0)
        {
         EventsArray[i].Surprise = ((EventsArray[i].Actual - EventsArray[i].Forecast) / MathAbs(EventsArray[i].Forecast)) * 100;

         if(EventsArray[i].Surprise > 0.5)
            EventsArray[i].Deviation = "Better";
         else
            if(EventsArray[i].Surprise < -0.5)
               EventsArray[i].Deviation = "Worse";
            else
               EventsArray[i].Deviation = "Neutral";
        }
      else
        {
         EventsArray[i].Surprise = 0;
         EventsArray[i].Deviation = "Neutral";
        }

      EventsArray[i].IsLive = (TimeCurrent() >= EventsArray[i].EventTime);
     }

// Sort events by time
   SortEventsByTime();
  }

//+------------------------------------------------------------------+
//| Generate event description                                       |
//+------------------------------------------------------------------+
string GenerateEventDescription(string currency, string title, double previous, double forecast, double actual)
  {
   string description = "";

   if(title == "Interest Rate Decision")
     {
      description = "The " + currency + " central bank's decision on the benchmark interest rate. ";
      description += "Higher rates typically strengthen the currency by attracting foreign investment. ";
      description += "Previous: " + DoubleToString(previous, 2) + "%, Forecast: " + DoubleToString(forecast, 2) + "%, Actual: " + DoubleToString(actual, 2) + "%.";
     }
   else
      if(title == "CPI (Consumer Price Index)")
        {
         description = "Measures the change in prices of goods and services purchased by consumers. ";
         description += "A higher than expected reading is positive for the " + currency + " currency. ";
         description += "Previous: " + DoubleToString(previous, 2) + "%, Forecast: " + DoubleToString(forecast, 2) + "%, Actual: " + DoubleToString(actual, 2) + "%.";
        }
      else
         if(title == "Non-Farm Payrolls")
           {
            description = "The change in the number of employed people during the previous month, excluding the farming industry. ";
            description += "A higher than expected reading is positive for the " + currency + " currency. ";
            description += "Previous: " + DoubleToString(previous, 2) + "K, Forecast: " + DoubleToString(forecast, 2) + "K, Actual: " + DoubleToString(actual, 2) + "K.";
           }
         else
            if(title == "Unemployment Rate")
              {
               description = "The percentage of the total workforce that is unemployed and actively seeking employment. ";
               description += "A lower than expected reading is positive for the " + currency + " currency. ";
               description += "Previous: " + DoubleToString(previous, 2) + "%, Forecast: " + DoubleToString(forecast, 2) + "%, Actual: " + DoubleToString(actual, 2) + "%.";
              }
            else
               if(title == "GDP (Gross Domestic Product)")
                 {
                  description = "The total value of all goods and services produced by the economy. ";
                  description += "A higher than expected reading is positive for the " + currency + " currency. ";
                  description += "Previous: " + DoubleToString(previous, 2) + "%, Forecast: " + DoubleToString(forecast, 2) + "%, Actual: " + DoubleToString(actual, 2) + "%.";
                 }
               else
                  if(title == "Retail Sales")
                    {
                     description = "Measures the change in the total value of sales at the retail level. ";
                     description += "A higher than expected reading is positive for the " + currency + " currency. ";
                     description += "Previous: " + DoubleToString(previous, 2) + "%, Forecast: " + DoubleToString(forecast, 2) + "%, Actual: " + DoubleToString(actual, 2) + "%.";
                    }
                  else
                     if(title == "Manufacturing PMI")
                       {
                        description = "Purchasing Managers' Index for the manufacturing sector. Above 50 indicates expansion. ";
                        description += "A higher than expected reading is positive for the " + currency + " currency. ";
                        description += "Previous: " + DoubleToString(previous, 2) + ", Forecast: " + DoubleToString(forecast, 2) + ", Actual: " + DoubleToString(actual, 2) + ".";
                       }
                     else
                        if(title == "Trade Balance")
                          {
                           description = "Difference between exports and imports of goods. A positive value indicates trade surplus. ";
                           description += "A higher than expected reading is positive for the " + currency + " currency. ";
                           description += "Previous: " + DoubleToString(previous, 2) + "B, Forecast: " + DoubleToString(forecast, 2) + "B, Actual: " + DoubleToString(actual, 2) + "B.";
                          }
                        else
                          {
                           description = "Economic indicator for " + currency + ". ";
                           description += "Previous: " + DoubleToString(previous, 2) + ", Forecast: " + DoubleToString(forecast, 2) + ", Actual: " + DoubleToString(actual, 2) + ".";
                          }

   return description;
  }

//+------------------------------------------------------------------+
//| Sort events by time                                              |
//+------------------------------------------------------------------+
void SortEventsByTime()
  {
   int size = ArraySize(EventsArray);
   for(int i=0; i<size-1; i++)
     {
      for(int j=i+1; j<size; j++)
        {
         if(EventsArray[i].EventTime > EventsArray[j].EventTime)
           {
            EconomicEvent temp = EventsArray[i];
            EventsArray[i] = EventsArray[j];
            EventsArray[j] = temp;
           }
        }
     }
  }

//+------------------------------------------------------------------+
//| Filter events based on settings                                  |
//+------------------------------------------------------------------+
bool ShouldShowEvent(EconomicEvent &event)
  {
// Show only events for current pair's currencies
   if(event.Currency != BaseCurrency && event.Currency != QuoteCurrency)
      return false;

// Filter by impact
   if(ShowHighImpactOnly && event.Impact != "High")
      return false;

// Filter by time
   if(event.EventTime > TimeCurrent() + (HoursAhead * 3600))
      return false;

// Don't show events older than 2 hours
   if(event.EventTime < TimeCurrent() - 7200)
      return false;

   return true;
  }

//+------------------------------------------------------------------+
//| Create main panel                                                |
//+------------------------------------------------------------------+
void CreateMainPanel()
  {
// Background panel
   ObjectCreate(0, "EconPanel_BG", OBJ_RECTANGLE_LABEL, 0, 0, 0);
   ObjectSetInteger(0, "EconPanel_BG", OBJPROP_XDISTANCE, PanelX);
   ObjectSetInteger(0, "EconPanel_BG", OBJPROP_YDISTANCE, PanelY);
   ObjectSetInteger(0, "EconPanel_BG", OBJPROP_XSIZE, PanelWidth);
   ObjectSetInteger(0, "EconPanel_BG", OBJPROP_YSIZE, PanelHeight);
   ObjectSetInteger(0, "EconPanel_BG", OBJPROP_BGCOLOR, clrDarkSlateGray);
   ObjectSetInteger(0, "EconPanel_BG", OBJPROP_BORDER_COLOR, clrBlack);
   ObjectSetInteger(0, "EconPanel_BG", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "EconPanel_BG", OBJPROP_BACK, false);
   ObjectSetInteger(0, "EconPanel_BG", OBJPROP_SELECTABLE, false);

// Header
   ObjectCreate(0, "EconPanel_Header", OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, "EconPanel_Header", OBJPROP_XDISTANCE, PanelX + 10);
   ObjectSetInteger(0, "EconPanel_Header", OBJPROP_YDISTANCE, PanelY + 10);
   ObjectSetInteger(0, "EconPanel_Header", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "EconPanel_Header", OBJPROP_FONTSIZE, 10);
   ObjectSetString(0, "EconPanel_Header", OBJPROP_TEXT, "Economic Calendar for " + Symbol());
   ObjectSetInteger(0, "EconPanel_Header", OBJPROP_COLOR, clrWhite);
   ObjectSetInteger(0, "EconPanel_Header", OBJPROP_BACK, false);

// Time label
   ObjectCreate(0, "EconPanel_Time", OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, "EconPanel_Time", OBJPROP_XDISTANCE, PanelX + 10);
   ObjectSetInteger(0, "EconPanel_Time", OBJPROP_YDISTANCE, PanelY + PanelHeight - 20);
   ObjectSetInteger(0, "EconPanel_Time", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "EconPanel_Time", OBJPROP_FONTSIZE, 8);
   ObjectSetString(0, "EconPanel_Time", OBJPROP_TEXT, "Last Update: Never");
   ObjectSetInteger(0, "EconPanel_Time", OBJPROP_COLOR, clrLightGray);
   ObjectSetInteger(0, "EconPanel_Time", OBJPROP_BACK, false);

// Refresh button
   ObjectCreate(0, "EconPanel_Refresh", OBJ_BUTTON, 0, 0, 0);
   ObjectSetInteger(0, "EconPanel_Refresh", OBJPROP_XDISTANCE, PanelX + PanelWidth - 80);
   ObjectSetInteger(0, "EconPanel_Refresh", OBJPROP_YDISTANCE, PanelY + 10);
   ObjectSetInteger(0, "EconPanel_Refresh", OBJPROP_XSIZE, 70);
   ObjectSetInteger(0, "EconPanel_Refresh", OBJPROP_YSIZE, 20);
   ObjectSetString(0, "EconPanel_Refresh", OBJPROP_TEXT, "Refresh");
   ObjectSetInteger(0, "EconPanel_Refresh", OBJPROP_BGCOLOR, clrDodgerBlue);
   ObjectSetInteger(0, "EconPanel_Refresh", OBJPROP_COLOR, clrWhite);
   ObjectSetInteger(0, "EconPanel_Refresh", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "EconPanel_Refresh", OBJPROP_BACK, false);

   PanelInitialized = true;
  }

//+------------------------------------------------------------------+
//| Update panel display                                             |
//+------------------------------------------------------------------+
void UpdatePanelDisplay()
  {
   if(!PanelInitialized)
      return;

// Clear existing event objects
   ClearEventDisplay();

// Create table headers
   CreateTableHeaders();

   int displayedEvents = 0;
   PendingEvents = 0;
   int liveEvents = 0;
   int baseCount = 0;
   int quoteCount = 0;

   for(int i=0; i<ArraySize(EventsArray); i++)
     {
      if(ShouldShowEvent(EventsArray[i]))
        {
         displayedEvents++;

         if(EventsArray[i].EventTime > TimeCurrent())
            PendingEvents++;
         else
            if(EventsArray[i].IsLive)
               liveEvents++;

         if(EventsArray[i].Currency == BaseCurrency)
            baseCount++;
         else
            if(EventsArray[i].Currency == QuoteCurrency)
               quoteCount++;

         DisplayEventRow(displayedEvents, i, EventsArray[i]);
        }
     }

   TotalEvents = displayedEvents;

// Update header
   string headerText = Symbol() + " Calendar (" + BaseCurrency + ": " + IntegerToString(baseCount);
   headerText += ", " + QuoteCurrency + ": " + IntegerToString(quoteCount) + ")";
   if(liveEvents > 0)
      headerText += " - " + IntegerToString(liveEvents) + " Live";

   ObjectSetString(0, "EconPanel_Header", OBJPROP_TEXT, headerText);
   ObjectSetString(0, "EconPanel_Time", OBJPROP_TEXT, "Last Update: " + TimeToString(TimeCurrent(), TIME_MINUTES|TIME_SECONDS));
  }

//+------------------------------------------------------------------+
//| Create table headers                                             |
//+------------------------------------------------------------------+
void CreateTableHeaders()
  {
   string headers[8] = {"Time", "Curr", "Event", "Impact", "Prev", "Fcst", "Actual", "Details"};
   int widths[8] = {70, 50, 170, 60, 70, 70, 70, 80};

   for(int i=0; i<8; i++)
     {
      string objName = "EconPanel_Header_" + IntegerToString(i);
      ObjectCreate(0, objName, OBJ_LABEL, 0, 0, 0);
      ObjectSetInteger(0, objName, OBJPROP_XDISTANCE, PanelX + 10 + GetColumnX(i, widths));
      ObjectSetInteger(0, objName, OBJPROP_YDISTANCE, PanelY + 40);
      ObjectSetInteger(0, objName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
      ObjectSetInteger(0, objName, OBJPROP_FONTSIZE, 8);
      ObjectSetString(0, objName, OBJPROP_TEXT, headers[i]);
      ObjectSetInteger(0, objName, OBJPROP_COLOR, clrWhite);
      ObjectSetInteger(0, objName, OBJPROP_SELECTABLE, false);
      ObjectSetInteger(0, objName, OBJPROP_BACK, false);
     }
  }

//+------------------------------------------------------------------+
//| Get column X position                                            |
//+------------------------------------------------------------------+
int GetColumnX(int colIndex, int &widths[])
  {
   int pos = 0;
   for(int i=0; i<colIndex; i++)
      pos += widths[i];
   return pos;
  }

//+------------------------------------------------------------------+
//| Display event row                                                |
//+------------------------------------------------------------------+
void DisplayEventRow(int rowIndex, int eventIndex, EconomicEvent &event)
  {
   int widths[8] = {70, 50, 170, 60, 70, 70, 70, 80};
   int startY = PanelY + 60;
   int rowHeight = 20;

   string timeStr = TimeToString(event.EventTime, TIME_MINUTES);
   string prevStr = (event.Previous != 0) ? DoubleToString(event.Previous, 2) : "N/A";
   string fcstStr = (event.Forecast != 0) ? DoubleToString(event.Forecast, 2) : "N/A";
   string actStr = (event.Actual != 0) ? DoubleToString(event.Actual, 2) : "N/A";

   if(event.IsLive)
      timeStr = "▶ " + timeStr;

// Text values for columns
   string values[7] = {timeStr, event.Currency, event.Title, event.Impact, prevStr, fcstStr, actStr};

// Display text columns
   for(int i=0; i<7; i++)
     {
      string objName = "EconPanel_Row_" + IntegerToString(rowIndex) + "_Col_" + IntegerToString(i);
      ObjectCreate(0, objName, OBJ_LABEL, 0, 0, 0);
      ObjectSetInteger(0, objName, OBJPROP_XDISTANCE, PanelX + 10 + GetColumnX(i, widths));
      ObjectSetInteger(0, objName, OBJPROP_YDISTANCE, startY + (rowIndex * rowHeight));
      ObjectSetInteger(0, objName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
      ObjectSetInteger(0, objName, OBJPROP_FONTSIZE, 8);
      ObjectSetString(0, objName, OBJPROP_TEXT, values[i]);

      color textColor = clrWhite;
      if(event.Impact == "High")
         textColor = ColorHighImpact;
      else
         if(event.Impact == "Medium")
            textColor = ColorMediumImpact;
         else
            textColor = ColorLowImpact;

      if(event.IsLive)
        {
         ObjectSetInteger(0, objName, OBJPROP_COLOR, clrYellow);
        }
      else
         if(i == 6) // Actual column
           {
            if(event.Deviation == "Better")
               ObjectSetInteger(0, objName, OBJPROP_COLOR, ColorPositive);
            else
               if(event.Deviation == "Worse")
                  ObjectSetInteger(0, objName, OBJPROP_COLOR, ColorNegative);
               else
                  ObjectSetInteger(0, objName, OBJPROP_COLOR, textColor);
           }
         else
           {
            ObjectSetInteger(0, objName, OBJPROP_COLOR, textColor);
           }

      ObjectSetInteger(0, objName, OBJPROP_SELECTABLE, false);
      ObjectSetInteger(0, objName, OBJPROP_BACK, false);
     }

// Create Details button
   string btnName = "EconPanel_DetailsBtn_" + IntegerToString(rowIndex) + "_" + IntegerToString(eventIndex);
   ObjectCreate(0, btnName, OBJ_BUTTON, 0, 0, 0);
   ObjectSetInteger(0, btnName, OBJPROP_XDISTANCE, PanelX + 10 + GetColumnX(7, widths) + 5);
   ObjectSetInteger(0, btnName, OBJPROP_YDISTANCE, startY + (rowIndex * rowHeight) - 2);
   ObjectSetInteger(0, btnName, OBJPROP_XSIZE, 70);
   ObjectSetInteger(0, btnName, OBJPROP_YSIZE, 18);
   ObjectSetString(0, btnName, OBJPROP_TEXT, "Details");
   ObjectSetInteger(0, btnName, OBJPROP_BGCOLOR, clrSteelBlue);
   ObjectSetInteger(0, btnName, OBJPROP_COLOR, clrWhite);
   ObjectSetInteger(0, btnName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, btnName, OBJPROP_FONTSIZE, 8);
   ObjectSetInteger(0, btnName, OBJPROP_BACK, false);
  }

//+------------------------------------------------------------------+
//| Clear event display                                              |
//+------------------------------------------------------------------+
void ClearEventDisplay()
  {
   int totalObjects = ObjectsTotal(0);
   for(int i=totalObjects-1; i>=0; i--)
     {
      string name = ObjectName(0, i);
      if(StringFind(name, "EconPanel_Row_") >= 0 ||
         StringFind(name, "EconPanel_Header_") >= 0 ||
         StringFind(name, "EconPanel_DetailsBtn_") >= 0)
        {
         ObjectDelete(0, name);
        }
     }
  }

//+------------------------------------------------------------------+
//| Show event details panel                                         |
//+------------------------------------------------------------------+
void ShowEventDetails(int eventIndex)
  {
   if(eventIndex < 0 || eventIndex >= ArraySize(EventsArray))
      return;

   SelectedEventIndex = eventIndex;
   EconomicEvent event = EventsArray[eventIndex];

// Create details panel background - SOLID OPAQUE
   ObjectCreate(0, "DetailsPanel_BG", OBJ_RECTANGLE_LABEL, 0, 0, 0);
   ObjectSetInteger(0, "DetailsPanel_BG", OBJPROP_XDISTANCE, PanelX + 100);
   ObjectSetInteger(0, "DetailsPanel_BG", OBJPROP_YDISTANCE, PanelY + 100);
   ObjectSetInteger(0, "DetailsPanel_BG", OBJPROP_XSIZE, 500);
   ObjectSetInteger(0, "DetailsPanel_BG", OBJPROP_YSIZE, 300);
   ObjectSetInteger(0, "DetailsPanel_BG", OBJPROP_BGCOLOR, C'30,30,30'); // Solid dark gray - NOT TRANSPARENT
   ObjectSetInteger(0, "DetailsPanel_BG", OBJPROP_BORDER_COLOR, clrGold);

   ObjectSetInteger(0, "DetailsPanel_BG", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "DetailsPanel_BG", OBJPROP_BACK, false); // Draw in foreground
   ObjectSetInteger(0, "DetailsPanel_BG", OBJPROP_SELECTABLE, false);
   ObjectSetInteger(0, "DetailsPanel_BG", OBJPROP_ZORDER, 1000); // Highest z-order

// Title
   ObjectCreate(0, "DetailsPanel_Title", OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, "DetailsPanel_Title", OBJPROP_XDISTANCE, PanelX + 110);
   ObjectSetInteger(0, "DetailsPanel_Title", OBJPROP_YDISTANCE, PanelY + 110);
   ObjectSetInteger(0, "DetailsPanel_Title", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "DetailsPanel_Title", OBJPROP_FONTSIZE, 10);
   ObjectSetString(0, "DetailsPanel_Title", OBJPROP_TEXT, event.Title + " (" + event.Currency + ")");
   ObjectSetInteger(0, "DetailsPanel_Title", OBJPROP_COLOR, clrGold);
   ObjectSetInteger(0, "DetailsPanel_Title", OBJPROP_BACK, false);
   ObjectSetInteger(0, "DetailsPanel_Title", OBJPROP_ZORDER, 1001);

// Time
   ObjectCreate(0, "DetailsPanel_Time", OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, "DetailsPanel_Time", OBJPROP_XDISTANCE, PanelX + 110);
   ObjectSetInteger(0, "DetailsPanel_Time", OBJPROP_YDISTANCE, PanelY + 135);
   ObjectSetInteger(0, "DetailsPanel_Time", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "DetailsPanel_Time", OBJPROP_FONTSIZE, 9);
   ObjectSetString(0, "DetailsPanel_Time", OBJPROP_TEXT, "Time: " + TimeToString(event.EventTime));
   ObjectSetInteger(0, "DetailsPanel_Time", OBJPROP_COLOR, clrWhite);
   ObjectSetInteger(0, "DetailsPanel_Time", OBJPROP_BACK, false);
   ObjectSetInteger(0, "DetailsPanel_Time", OBJPROP_ZORDER, 1001);

// Impact
   ObjectCreate(0, "DetailsPanel_Impact", OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, "DetailsPanel_Impact", OBJPROP_XDISTANCE, PanelX + 110);
   ObjectSetInteger(0, "DetailsPanel_Impact", OBJPROP_YDISTANCE, PanelY + 155);
   ObjectSetInteger(0, "DetailsPanel_Impact", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "DetailsPanel_Impact", OBJPROP_FONTSIZE, 9);
   ObjectSetString(0, "DetailsPanel_Impact", OBJPROP_TEXT, "Impact: " + event.Impact);

// Set impact color
   color impactColor = clrWhite;
   if(event.Impact == "High")
      impactColor = ColorHighImpact;
   else
      if(event.Impact == "Medium")
         impactColor = ColorMediumImpact;
      else
         impactColor = ColorLowImpact;
   ObjectSetInteger(0, "DetailsPanel_Impact", OBJPROP_COLOR, impactColor);
   ObjectSetInteger(0, "DetailsPanel_Impact", OBJPROP_BACK, false);
   ObjectSetInteger(0, "DetailsPanel_Impact", OBJPROP_ZORDER, 1001);

// Values
   ObjectCreate(0, "DetailsPanel_Values", OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, "DetailsPanel_Values", OBJPROP_XDISTANCE, PanelX + 110);
   ObjectSetInteger(0, "DetailsPanel_Values", OBJPROP_YDISTANCE, PanelY + 175);
   ObjectSetInteger(0, "DetailsPanel_Values", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "DetailsPanel_Values", OBJPROP_FONTSIZE, 9);
   ObjectSetString(0, "DetailsPanel_Values", OBJPROP_TEXT, "Previous: " + DoubleToString(event.Previous, 2) +
                   " | Forecast: " + DoubleToString(event.Forecast, 2) +
                   " | Actual: " + DoubleToString(event.Actual, 2));
   ObjectSetInteger(0, "DetailsPanel_Values", OBJPROP_COLOR, clrWhite);
   ObjectSetInteger(0, "DetailsPanel_Values", OBJPROP_BACK, false);
   ObjectSetInteger(0, "DetailsPanel_Values", OBJPROP_ZORDER, 1001);

// Deviation
   ObjectCreate(0, "DetailsPanel_Deviation", OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, "DetailsPanel_Deviation", OBJPROP_XDISTANCE, PanelX + 110);
   ObjectSetInteger(0, "DetailsPanel_Deviation", OBJPROP_YDISTANCE, PanelY + 195);
   ObjectSetInteger(0, "DetailsPanel_Deviation", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "DetailsPanel_Deviation", OBJPROP_FONTSIZE, 9);
   ObjectSetString(0, "DetailsPanel_Deviation", OBJPROP_TEXT, "Deviation: " + event.Deviation +
                   " (Surprise: " + DoubleToString(event.Surprise, 2) + "%)");

// Set deviation color
   color deviationColor = clrWhite;
   if(event.Deviation == "Better")
      deviationColor = ColorPositive;
   else
      if(event.Deviation == "Worse")
         deviationColor = ColorNegative;
   ObjectSetInteger(0, "DetailsPanel_Deviation", OBJPROP_COLOR, deviationColor);
   ObjectSetInteger(0, "DetailsPanel_Deviation", OBJPROP_BACK, false);
   ObjectSetInteger(0, "DetailsPanel_Deviation", OBJPROP_ZORDER, 1001);

// Description (wrapped)
   int maxCharsPerLine = 60;
   string desc = event.Description;
   int descY = PanelY + 220;
   int lineCount = 0;

// Split description into multiple lines
   while(StringLen(desc) > 0 && lineCount < 5)
     {
      int lineLength = MathMin(maxCharsPerLine, StringLen(desc));
      string line = StringSubstr(desc, 0, lineLength);

      // Find last space in line for clean break
      if(lineLength == maxCharsPerLine && StringLen(desc) > lineLength)
        {
         int lastSpace = StringFindReverse(line, " ");
         if(lastSpace > 40) // Only break if we have a reasonable space
           {
            line = StringSubstr(desc, 0, lastSpace);
            desc = StringSubstr(desc, lastSpace + 1);
           }
         else
           {
            desc = StringSubstr(desc, lineLength);
           }
        }
      else
        {
         desc = "";
        }

      string objName = "DetailsPanel_Desc_" + IntegerToString(lineCount);

      ObjectCreate(0, objName, OBJ_LABEL, 0, 0, 0);
      ObjectSetInteger(0, objName, OBJPROP_XDISTANCE, PanelX + 110);
      ObjectSetInteger(0, objName, OBJPROP_YDISTANCE, descY);
      ObjectSetInteger(0, objName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
      ObjectSetInteger(0, objName, OBJPROP_FONTSIZE, 8);
      ObjectSetString(0, objName, OBJPROP_TEXT, line);
      ObjectSetInteger(0, objName, OBJPROP_COLOR, clrLightGray);
      ObjectSetInteger(0, objName, OBJPROP_SELECTABLE, false);
      ObjectSetInteger(0, objName, OBJPROP_BACK, false);
      ObjectSetInteger(0, objName, OBJPROP_ZORDER, 1001);

      descY += 15;
      lineCount++;
     }

// Close button
   ObjectCreate(0, "DetailsPanel_Close", OBJ_BUTTON, 0, 0, 0);
   ObjectSetInteger(0, "DetailsPanel_Close", OBJPROP_XDISTANCE, PanelX + 110);
   ObjectSetInteger(0, "DetailsPanel_Close", OBJPROP_YDISTANCE, PanelY + 370);
   ObjectSetInteger(0, "DetailsPanel_Close", OBJPROP_XSIZE, 100);
   ObjectSetInteger(0, "DetailsPanel_Close", OBJPROP_YSIZE, 20);
   ObjectSetString(0, "DetailsPanel_Close", OBJPROP_TEXT, "Close");
   ObjectSetInteger(0, "DetailsPanel_Close", OBJPROP_BGCOLOR, clrFireBrick);
   ObjectSetInteger(0, "DetailsPanel_Close", OBJPROP_COLOR, clrWhite);
   ObjectSetInteger(0, "DetailsPanel_Close", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "DetailsPanel_Close", OBJPROP_BACK, false);
   ObjectSetInteger(0, "DetailsPanel_Close", OBJPROP_ZORDER, 1001);

   DetailsPanelVisible = true;
  }

//+------------------------------------------------------------------+
//| Find last occurrence of a substring                              |
//+------------------------------------------------------------------+
int StringFindReverse(string text, string search)
  {
   int pos = -1;
   for(int i=StringLen(text)-StringLen(search); i>=0; i--)
     {
      if(StringSubstr(text, i, StringLen(search)) == search)
        {
         pos = i;
         break;
        }
     }
   return pos;
  }

//+------------------------------------------------------------------+
//| Hide event details panel                                         |
//+------------------------------------------------------------------+
void HideEventDetails()
  {
// Delete all details panel objects
   int totalObjects = ObjectsTotal(0);
   for(int i=totalObjects-1; i>=0; i--)
     {
      string name = ObjectName(0, i);
      if(StringFind(name, "DetailsPanel_") >= 0)
        {
         ObjectDelete(0, name);
        }
     }

   DetailsPanelVisible = false;
   SelectedEventIndex = -1;
  }

//+------------------------------------------------------------------+
//| Extract event index from button name                             |
//+------------------------------------------------------------------+
int ExtractEventIndexFromButton(string buttonName)
  {
// Button name format: "EconPanel_DetailsBtn_X_Y" where X is row index, Y is event index
// We need to get the last number after the last underscore
   int lastUnderscore = StringFindReverse(buttonName, "_");
   if(lastUnderscore > 0)
     {
      string indexStr = StringSubstr(buttonName, lastUnderscore + 1);
      // StringToInteger returns long, so we need to cast to int
      return (int)StringToInteger(indexStr);
     }
   return -1;
  }

//+------------------------------------------------------------------+
//| Refresh data                                                     |
//+------------------------------------------------------------------+
void RefreshData()
  {
   GenerateDemoData();
   IsConnected = true;

   if(IsConnected)
     {
      UpdatePanelDisplay();
      LastRefreshTime = TimeCurrent();
     }
  }

//+------------------------------------------------------------------+
//| Timer function                                                   |
//+------------------------------------------------------------------+
void OnTimer()
  {
   if(TimeCurrent() - LastRefreshTime >= RefreshInterval)
     {
      RefreshData();
     }
  }

//+------------------------------------------------------------------+
//| Chart event handler                                              |
//+------------------------------------------------------------------+
void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam)
  {
   if(id == CHARTEVENT_OBJECT_CLICK)
     {
      if(sparam == "EconPanel_Refresh")
        {
         RefreshData();
        }
      else
         if(sparam == "DetailsPanel_Close")
           {
            HideEventDetails();
           }
         else
            if(StringFind(sparam, "EconPanel_DetailsBtn_") >= 0)
              {
               // Extract event index from button name
               int eventIndex = ExtractEventIndexFromButton(sparam);

               if(eventIndex >= 0)
                 {
                  if(DetailsPanelVisible)
                    {
                     HideEventDetails();
                    }

                  ShowEventDetails(eventIndex);
                 }
              }
     }
  }

//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
  {
   return(rates_total);
  }

//+------------------------------------------------------------------+
//| Indicator initialization function                                |
//+------------------------------------------------------------------+
int OnInit()
  {
// Create main panel
   CreateMainPanel();

// Initial data refresh
   RefreshData();

// Set timer for updates
   EventSetTimer(60);

   return(INIT_SUCCEEDED);
  }

//+------------------------------------------------------------------+
//| Deinitialization function                                        |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   EventKillTimer();

// Delete all objects created by this indicator
   int totalObjects = ObjectsTotal(0);
   for(int i=totalObjects-1; i>=0; i--)
     {
      string name = ObjectName(0, i);
      if(StringFind(name, "EconPanel_") >= 0 || StringFind(name, "DetailsPanel_") >= 0)
        {
         ObjectDelete(0, name);
        }
     }
  }
//+------------------------------------------------------------------+
